#include "VertexShaderSemantics.fx"
#include "FragmentShaderSemantics.fx"

//-------------------------------------------------------------------------------------------------

uniform sampler2D layer0_sampler : register(s0); 
uniform sampler2D layer1_sampler : register(s1); 
uniform sampler2D layer2_sampler : register(s2); 
uniform sampler2D layer3_sampler : register(s3); 

#define g_ViewportScaleBias		vs_viewportScaleBias		// viewport scale xy, offset xy

uniform float4		ringParams1 : register(c132);
uniform float4		ringParams2 : register(c133);
uniform float3x3	rotMtx2D	: register(c134);

//-------------------------------------------------------------------------------------------------

struct VS_IN {
	float4 position		: POSITION;
	float2 uvSet0		: TEXCOORD0;
};

//-------------------------------------------------------------------------------------------------

struct RINGBLUR_VS_OUT { 
	float4 position		: POSITION;
	float4 texcoord		: TEXCOORD0;
	float4 ringParams1	: TEXCOORD1;	
	float4 color		: COLOR;
};

struct RINGBLUR_PS_IN {
	float4 texcoord		: TEXCOORD0;
	float4 ringParams1	: TEXCOORD1;	
	float4 color		: COLOR;
};

RINGBLUR_VS_OUT ringdarken_vx( VS_IN input )
{
	RINGBLUR_VS_OUT output;
	
	// ringParams1
	// x = uvOffset.x
	// y = uvOffset.y
	// z = alpha
	// w = colorScale

	output.position		= input.position;
	output.color		= float4( 1.0, 1.0, 1.0, 1.0 );
	output.ringParams1	= ringParams1;
	output.texcoord.xy	= input.uvSet0;
	output.texcoord.zw	= float2( 0.0, 0.0 );
	
	return output;
}

half4 ringdarken_px( RINGBLUR_PS_IN input ) : COLOR
{	
	half4 result;

	result.rgb = tex2D( layer0_sampler, input.texcoord.xy - input.ringParams1.xy ).rgb * input.ringParams1.w;
	result.a = input.ringParams1.z;
	
	return result;
}

//-------------------------------------------------------------------------------------------------

struct RINGEFFECT_VS_OUT { 
	float4 position		: POSITION;
	float4 texcoord		: TEXCOORD0;
	float4 ringParams1	: TEXCOORD1;
	float4 ringParams2	: TEXCOORD2;
	float4 color		: COLOR;
};

RINGEFFECT_VS_OUT ringeffect_vx( VS_IN input )
{
	RINGEFFECT_VS_OUT output;

	// ringParams1
	// x = time (used to rotate the noise map)
	// y = blur lod scale
	// z = centre.x
	// w = centre.x
	
	// ringParams2
	// x = normal amplitude
	// y = tangent amplitude
	
	// s0 = blurTex
	// s1 = noiseTex
	// s2 = accumTex
	// s3 = colorTex

	output.position		= input.position;
	output.color		= float4( 1.0, 1.0, 1.0, 1.0 );
	output.ringParams1	= ringParams1;
	output.ringParams2	= ringParams2;
	output.texcoord.xy	= input.uvSet0;
	
	output.texcoord.xy = output.texcoord.xy * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	// Transform the UV's to the correct position.
	half3 offset = half3( 0.5h, 0.5h, 0.0h );
	output.texcoord.zw = ( mul( half3( input.uvSet0, 1.0f ) - offset, rotMtx2D ) + offset ).xy;
			
	// And store a movement vector in here.
	output.ringParams1.zw = mul( half2( 1.0h, 0.0f ), rotMtx2D );
			
	return output;
}

//-------------------------------------------------------------------------------------------------

half4 ringeffect_px( RINGEFFECT_VS_OUT input ) : COLOR
{	
	half4 result;

	// Compute the vector from the blur center to the current pixel. 
	half2 blurNormal	= normalize( input.ringParams1.zw );
	half2 blurTangent	= float2( blurNormal.y, blurNormal.x );
		
	// Generate circular UV mapping for noise lookup.
	half2 noiseUV = input.texcoord.zw;

	// Generate an offset vector based on the noise map.
	half2 offset = tex2D( layer1_sampler, noiseUV ).rg * half2( 0.02, 0.02 );
	offset = blurNormal * offset.y * input.ringParams2.x + blurTangent * offset.x * input.ringParams2.y;
	
	// Lookup RGB in the accum texture (s).
	half3 accumVal = tex2D( layer2_sampler, input.texcoord.xy ).rgb;
#if (NUM_ACCUM_TEX > 1)
	accumVal += tex2D( layer3_sampler, input.texcoord.xy ).rgb;
#endif

	// Lookup RGB in the blur texture.
	half lod = ( accumVal.r + accumVal.g + accumVal.b ) * input.ringParams1.y;
	result.rgb = tex2Dlod( layer0_sampler, half4( input.texcoord.xy + offset, 0.0, lod ) ).rgb;

	// Blend in the accumulation pass.
	result.rgb += accumVal * 2.0;

	// Multiply in the colour.
	result.rgb *= input.color.rgb;
	
	// Alpha to 1 for now.
	result.a = 1.0f;

	return result;
}